<?php namespace WHITEHALLPLUGIN\Element;

use Elementor\Controls_Manager;
use Elementor\Controls_Stack;
use Elementor\Group_Control_Typography;
use Elementor\Scheme_Typography;
use Elementor\Scheme_Color;
use Elementor\Group_Control_Border;
use Elementor\Repeater;
use Elementor\Widget_Base;
use Elementor\Utils;
use Elementor\Group_Control_Text_Shadow;
use Elementor\Plugin;

/**
 * Elementor button widget.
 * Elementor widget that displays a button with the ability to control every
 * aspect of the button design.
 *
 * @since 1.0.0
 */
class Team_V2 extends Widget_Base {

    /**
     * Get widget name.
     * Retrieve button widget name.
     *
     * @since  1.0.0
     * @access public
     * @return string Widget name.
     */
    public function get_name() {
        return 'whitehall_team_v2';
    }

    /**
     * Get widget title.
     * Retrieve button widget title.
     *
     * @since  1.0.0
     * @access public
     * @return string Widget title.
     */
    public function get_title() {
        return esc_html__( 'Team V2', 'whitehall' );
    }

    /**
     * Get widget icon.
     * Retrieve button widget icon.
     *
     * @since  1.0.0
     * @access public
     * @return string Widget icon.
     */
    public function get_icon() {
        return 'fa fa-briefcase';
    }

    /**
     * Get widget categories.
     * Retrieve the list of categories the button widget belongs to.
     * Used to determine where to display the widget in the editor.
     *
     * @since  2.0.0
     * @access public
     * @return array Widget categories.
     */
    public function get_categories() {
        return [ 'whitehall' ];
    }

    /**
     * Register button widget controls.
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since  1.0.0
     * @access protected
     */
    protected function _register_controls() {
        $this->start_controls_section(
            'team_v2',
            [
                'label' => esc_html__( 'Team V2', 'whitehall' ),
            ]
        );
		$this->add_control(
            'bg_image',
            [
                'label' => __( 'Backgruond Image', 'whitehall' ),
                'type' => Controls_Manager::MEDIA,
                'default' => ['url' => Utils::get_placeholder_image_src(),],
            ]
        );
		$this->add_control(
            'subtitle',
            [
                'label'       => __( 'Sub Title', 'whitehall' ),
                'type'        => Controls_Manager::TEXT,
				'label_block' => true,
                'dynamic'     => [
                    'active' => true,
                ],
            ]
        );
        $this->add_control(
            'title',
            [
                'label'       => __( 'Title', 'whitehall' ),
                'type'        => Controls_Manager::TEXTAREA,
                'dynamic'     => [
                    'active' => true,
                ],
            ]
        );
        $this->add_control(
            'query_number',
            [
                'label'   => esc_html__( 'Number of post', 'whitehall' ),
                'type'    => Controls_Manager::NUMBER,
                'default' => 4,
                'min'     => 1,
                'max'     => 100,
                'step'    => 1,
            ]
        );
        $this->add_control(
            'query_orderby',
            [
                'label'   => esc_html__( 'Order By', 'whitehall' ),
                'type'    => Controls_Manager::SELECT,
                'default' => 'date',
                'options' => array(
                    'date'       => esc_html__( 'Date', 'whitehall' ),
                    'title'      => esc_html__( 'Title', 'whitehall' ),
                    'menu_order' => esc_html__( 'Menu Order', 'whitehall' ),
                    'rand'       => esc_html__( 'Random', 'whitehall' ),
                ),
            ]
        );
        $this->add_control(
            'query_order',
            [
                'label'   => esc_html__( 'Order', 'whitehall' ),
                'type'    => Controls_Manager::SELECT,
                'default' => 'ASC',
                'options' => array(
                    'DESC' => esc_html__( 'DESC', 'whitehall' ),
                    'ASC'  => esc_html__( 'ASC', 'whitehall' ),
                ),
            ]
        );
        $this->add_control(
            'query_category',
            [
                'type' => Controls_Manager::SELECT,
                'label' => esc_html__('Category', 'whitehall'),
                'options' => get_categories_list('team_cat')
            ]
        );
        $this->end_controls_section();
    }

    /**
     * Render button widget output on the frontend.
     * Written in PHP and used to generate the final HTML.
     *
     * @since  1.0.0
     * @access protected
     */
    protected function render() {
        $settings = $this->get_settings_for_display();
        $allowed_tags = wp_kses_allowed_html('post');

        $paged = whitehall_set($_POST, 'paged') ? esc_attr($_POST['paged']) : 1;

        $this->add_render_attribute( 'wrapper', 'class', 'templatepath-whitehall' );
        $args = array(
            'post_type'      => 'team',
            'posts_per_page' => whitehall_set( $settings, 'query_number' ),
            'orderby'        => whitehall_set( $settings, 'query_orderby' ),
            'order'          => whitehall_set( $settings, 'query_order' ),
            'paged'          => $paged
        );

        if( whitehall_set( $settings, 'query_category' ) ) $args['team_cat'] = whitehall_set( $settings, 'query_category' );
        $query = new \WP_Query( $args );

        if ( $query->have_posts() ) { ?>
        
		<!-- team-section -->
        <section class="team-section alternat-2 sec-pad pb-140">
            <div class="pattern-layer" style="background-image: url(<?php echo wp_get_attachment_url($settings['bg_image']['id']); ?>);"></div>
            <div class="auto-container">
                <div class="sec-title centred">
					<?php if($settings['subtitle']){ ?>
                    <h6><i class="flaticon-star"></i><span><?php echo wp_kses( $settings['subtitle'], true ); ?></span><i class="flaticon-star"></i></h6>
					<?php } ?>
					
					<?php if($settings['title']){ ?>
                    <h2><?php echo wp_kses( $settings['title'], true ); ?></h2>
                    <div class="title-shape"></div>
					<?php } ?>
                </div>
                <div class="four-item-carousel owl-carousel owl-theme owl-nav-none">
                    <?php while($query->have_posts()) : $query->the_post(); ?>
                    <div class="team-block-one">
                        <div class="inner-box">
                            <figure class="image-box"><?php the_post_thumbnail('whitehall_270x270'); ?></figure>
                            <div class="lower-content">
                                <div class="author-box">
                                    <h4><?php the_title(); ?></h4>
                                    <span class="designation"><?php echo get_post_meta( get_the_id(), 'designation', true ); ?></span>
                                </div>
                                <ul class="othre-info clearfix">
                                    <?php if(get_post_meta( get_the_id(), 'email', true )) { ?>
									<li class="mail-box"><a href="mailto:<?php echo sanitize_email(get_post_meta( get_the_id(), 'email', true )); ?>"><i class="flaticon-mail-inbox-app"></i><?php esc_html_e('Email', 'whitehall');  ?></a></li>
									<?php } ?>

									<?php if(get_post_meta( get_the_id(), 'phone', true )) { ?>
									<li class="phone-box"><i class="flaticon-emergency-call"></i><a href="tel:<?php echo esc_attr(phone_number(get_post_meta( get_the_id(), 'phone', true ))); ?>"><?php echo wp_kses(get_post_meta( get_the_id(), 'phone', true ), true); ?></a></li>
									<?php } ?>

									<?php $icons = get_post_meta( get_the_id(), 'social_profile', true );
										if ( ! empty( $icons ) ) :
									?>
									<li class="share-option">
										<i class="share-icon flaticon-share"></i>
										<ul class="share-links clearfix">
											<?php foreach ( $icons as $h_icon ) :
											$header_social_icons = json_decode( urldecode( whitehall_set( $h_icon, 'data' ) ) );
											if ( whitehall_set( $header_social_icons, 'enable' ) == '' ) {
												continue;
											}
											$icon_class = explode( '-', whitehall_set( $header_social_icons, 'icon' ) ); ?>
											<li><a href="<?php echo whitehall_set( $header_social_icons, 'url' ); ?>" style="background-color:<?php echo whitehall_set( $header_social_icons, 'background' ); ?>; color: <?php echo whitehall_set( $header_social_icons, 'color' ); ?>" target="_blank"><i class="fab <?php echo esc_attr( whitehall_set( $header_social_icons, 'icon' ) ); ?>"></i></a></li>
											<?php endforeach; ?>
										</ul>
									</li>
									<?php endif; ?>
                                </ul>
                            </div>
                        </div>
                    </div>
					<?php endwhile; ?>
                </div>
            </div>
        </section>
        <!-- team-section end -->

        <?php }

        wp_reset_postdata();
    }
}
